﻿#include  "StdAfx.h"
#include  "Resource.h"

#include  "Decompressor.hpp"
#include  "CommandProcessorState.hpp"
#include  "ProcessUI.hpp"
#include  "OverwriteDialog.hpp"

#include  <szFileArchiveVolume.hpp>
#include  <szArchiveHandler.hpp>
#include  <szArchiveContent.hpp>
#include  <szArchiveEnumerator.hpp>
#include  <szArchiveExtractor.hpp>
#include  <szStoredItem.hpp>
#include  <szStoredItemContainer.hpp>
#include  <szRuntimeException.hpp>
#include  <szPath.hpp>
#include  <szFile.hpp>
#include  <szStdio.hpp>

#include  <boost/algorithm/string.hpp>
#include  <cstdio>
#include  <zlib.h>
#include  <bzlib.h>

SZ_AN_BEG

const u64 SEARCH_LIMIT_SIZE = 1024 * 1024; // 1MB
const int LIB_BUF_SIZE = 256 * 1024;

class inflateScope
{
private:
  z_stream *m_strm;
public:
  inflateScope(z_stream *strm) : m_strm(strm) {}
  ~inflateScope() { inflateEnd(m_strm); }
};

class bzipScope
{
private:
  bz_stream *m_strm;
public:
  bzipScope(bz_stream *strm) : m_strm(strm) {}
  ~bzipScope() { BZ2_bzDecompressEnd(m_strm); }
};

inline void GzipFail()
{
  BOOST_THROW_EXCEPTION(szpp::RuntimeException(SZT("Cannot decompress gzip stream")));
}

inline void Bzip2Fail()
{
  BOOST_THROW_EXCEPTION(szpp::RuntimeException(SZT("Cannot decompress bzip2 stream")));
}

SZ_AN_END

using namespace std;
using namespace szpp;

Decompressor::Decompressor(const szstring &archivePath, CommandProcessorState *pState, ProcessUI *pUI)
: archivePath(archivePath), virtualPath(), realPath(),
  password(), pState(pState), pUI(pUI), archiveContent(), canceled(false)
{
}

Decompressor::~Decompressor()
{
  if (!realPath.empty() && FileExists(realPath))
    DeleteFile(realPath.c_str());
}

#pragma warning(push)
#pragma warning(disable: 4996)

void Decompressor::DecompressGzip(const szstring &srcPath, const szstring &dstPath)
{
  u64 totSize = GetFileSize(srcPath), curSize = 0;
  ScopedFile ifp(_tfopen(srcPath.c_str(), SZL("rb"))), ofp(_tfopen(dstPath.c_str(), SZL("wb")));

  hbuf<u08, LIB_BUF_SIZE> ibuf, obuf;
  z_stream strm;

  strm.next_in = 0;
  strm.avail_in = 0;
  strm.zalloc = Z_NULL;
  strm.zfree = Z_NULL;
  strm.opaque = Z_NULL;
  int ret = inflateInit2(&strm, 15 + 32 /* auto-detect zlib/gzip */);
  if (ret != Z_OK)
    GzipFail();

  inflateScope inf_scope(&strm);

  pUI->SetItem(SZT("Decompressing gzip stream..."), SZL_EMPTY);
  SetTotal(0, totSize);
  do
  {
    strm.avail_in = (uInt)fread(ibuf, 1, ibuf.size(), ifp);
    curSize += strm.avail_in;
    if (ferror(ifp))
      GzipFail();
    if (strm.avail_in == 0)
      break;
    strm.next_in = ibuf;
    do
    {
      strm.avail_out = obuf.size();
      strm.next_out = obuf;
      ret = inflate(&strm, Z_NO_FLUSH);
      if (ret != Z_OK && ret != Z_STREAM_END)
        GzipFail();

      u32 have = obuf.size() - strm.avail_out;
      if (fwrite(obuf, 1, have, ofp) != have || ferror(ofp))
        GzipFail();
    } while(strm.avail_out == 0);

    SetCompleted(0, curSize);
  } while (ret != Z_STREAM_END);
}

void Decompressor::DecompressBzip2(const szstring &srcPath, const szstring &dstPath)
{
  u64 totSize = GetFileSize(srcPath), curSize = 0;
  ScopedFile ifp(_tfopen(srcPath.c_str(), SZL("rb"))), ofp(_tfopen(dstPath.c_str(), SZL("wb")));

  hbuf<char, LIB_BUF_SIZE> ibuf, obuf;
  bz_stream strm;
  memset(&strm, 0, sizeof(strm));

  strm.next_in = 0;
  strm.avail_in = 0;
  int ret = BZ2_bzDecompressInit(&strm, 0, 0);
  if (ret != BZ_OK)
    Bzip2Fail();

  bzipScope bz_scope(&strm);

  pUI->SetItem(SZT("Decompressing bzip2 stream..."), SZL_EMPTY);
  SetTotal(0, totSize);
  do
  {
    strm.avail_in = (uInt)fread(ibuf, 1, ibuf.size(), ifp);
    curSize += strm.avail_in;
    if (ferror(ifp))
      Bzip2Fail();
    if (strm.avail_in == 0)
      break;
    strm.next_in = ibuf;
    do
    {
      strm.avail_out = obuf.size();
      strm.next_out = obuf;
      ret = BZ2_bzDecompress(&strm);
      if (ret != BZ_OK && ret != BZ_STREAM_END)
        Bzip2Fail();

      u32 have = obuf.size() - strm.avail_out;
      if (fwrite(obuf, 1, have, ofp) != have || ferror(ofp))
        Bzip2Fail();
    } while(strm.avail_out == 0);

    SetCompleted(0, curSize);
  } while (ret != BZ_STREAM_END);
}

#pragma warning(pop)

void Decompressor::PreprocessTar()
{
  const szstring extension = boost::algorithm::to_lower_copy(ExtractExtension(archivePath));
  const szstring stemExtension = boost::algorithm::to_lower_copy(ExtractExtension(ExtractStem(archivePath)));

  bool isGzip = false, isBzip2 = false; // tar + gzip または tar + bzip2 っぽい場合に true となる（i.e. 単なる gzip, bzip2 では true にならない）

  if (extension == SZL("tgz"))
  {
    isGzip = true;
    virtualPath = ChangeExtension(archivePath, SZL("tar"));
  }
  else if (extension == SZL("gz") && stemExtension == SZL("tar"))
  {
    isGzip = true;
    virtualPath = RemoveExtension(archivePath);
  }
  else if (extension == SZL("tbz") || extension == SZL("tbz2"))
  {
    isBzip2 = true;
    virtualPath = ChangeExtension(archivePath, SZL("tar"));
  }
  else if (extension == SZL("bz2") || extension == SZL("bzip2"))
  {
    isBzip2 = true;
    virtualPath = RemoveExtension(archivePath);
  }
  else
    return;

  const szstring tempFileName = CreateTempFileName(ExtractDirectory(virtualPath));
  if (FileExists(tempFileName))
    DeleteFile(tempFileName.c_str());

  realPath = ChangeExtension(tempFileName, SZL("tar"));
  if (isGzip)
  {
    DecompressGzip(archivePath, realPath);
  }
  else
  {
    DecompressBzip2(archivePath, realPath);
  }
}

void Decompressor::Open()
{
  // tar + gzip/bzip2 と思われるファイルは一時ファイルに展開した上で処理する
  PreprocessTar();

  pUI->SetItem(SZT("Reading archive file..."), SZL_EMPTY);

  canceled = false;

  FileArchiveVolume archiveVolume(realPath.empty() ? archivePath : realPath);
  ArchiveHandler *handler;
  archiveContent.reset(ArchiveEnumerator::EnumerateUnknown(&handler, &archiveVolume, SEARCH_LIMIT_SIZE, this, this));
  ThrowIfCanceled();
  if (handler == 0)
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Cannot create handler")));
  if (archiveContent.get() == 0)
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Cannot open archive")));
}

void Decompressor::Decompress()
{
  szstring outputDir = !pState->GetOutputPathRoot().empty() ? pState->GetOutputPathRoot() : ExtractDirectory(archivePath);

  if (archiveContent->GetRoot()->GetNumberOfSubContainers() + archiveContent->GetRoot()->GetNumberOfSubItems() > 1)
    outputDir = Combine(outputDir, ExtractStem(virtualPath.empty() ? archivePath : virtualPath));

  ArchiveExtractor::Extract(realPath.empty() ? archivePath : realPath, outputDir, password.empty() ? NULL : &password, this, this, this);
  ThrowIfCanceled();
}

//
// ArchiveOpenClientCallback
//

HRESULT Decompressor::SetTotal(const u64 &files, const u64 &bytes)
{
  pUI->SetMaximum(files != 0 ? files : bytes);
  return CheckQuitAndReturn();
}

HRESULT Decompressor::SetCompleted(const u64 &files, const u64 &bytes)
{
  pUI->SetCurrent(files != 0 ? files : bytes);
  return CheckQuitAndReturn();
}

//
// GetPasswordClientCallback
//

HRESULT Decompressor::GetPassword(szstring *password)
{
  switch (pState->GetPassword(archivePath, password))
  {
  case PasswordResult::PASSWORD_CANCEL:
    return E_ABORT;
  case PasswordResult::PASSWORD_SKIP:
    return S_FALSE;
  }
  this->password.assign(*password);
  return S_OK;
}

//
// ArchiveFileExtractClientCallback
//

HRESULT Decompressor::SetTotal(const u64 &total)
{
  pUI->SetMaximum(total);
  return CheckQuitAndReturn();
}

HRESULT Decompressor::SetCompleted(const u64 &completeValue)
{
  pUI->SetCurrent(completeValue);
  return CheckQuitAndReturn();
}

HRESULT Decompressor::CryptoGetTextPassword(szstring *password)
{
  return GetPassword(password);
}

LRESULT AskOverwriteFunc(void * ptr)
{
  COverwriteDialog * dlg = (COverwriteDialog *)ptr;
  return dlg->DoModal();
}

HRESULT Decompressor::AskOverwrite(
    const szstring &existingName, const szpp::Time &existingTime, const u64 &existingSize,
    const szstring &newName, const szpp::Time &newTime, const u64 &newSize,
    u32 *answer)
{
  COverwriteDialog dialog(SZT("Overwrite the existing file?"), existingName, existingTime, existingSize, newTime, newSize, answer);
  pUI->Invoke(AskOverwriteFunc, &dialog);
  return S_OK;
}

HRESULT Decompressor::PrepareOperation(const szstring &name, bool isFolder, u32 extractMode, const u64 &position)
{
  pUI->SetItem(ExtractDirectory(name), ExtractFileName(name));
  return CheckQuitAndReturn();
}

HRESULT Decompressor::MessageError(const szstring &message)
{
  pState->AddLog(message);
  return S_OK;
}

HRESULT Decompressor::SetOperationResult(u32 operationResult, bool encrypted)
{
  return S_OK;
}

HRESULT Decompressor::BeforeOpen(const szstring &name)
{
  pUI->SetItem(ExtractDirectory(name), ExtractFileName(name));
  return CheckQuitAndReturn();
}

HRESULT Decompressor::OpenResult(const szstring &name, HRESULT result, bool encrypted)
{
  return S_OK;
}

HRESULT Decompressor::ThereAreNoFiles()
{
  return S_OK;
}

HRESULT Decompressor::ExtractResult(HRESULT result)
{
  return S_OK;
}

HRESULT Decompressor::SetPassword(const szstring &password)
{
  return S_OK;
}

HRESULT Decompressor::CheckQuitAndReturn()
{
  if (pState->Quitting())
  {
    canceled = true;
    return E_ABORT;
  }
  return S_OK;
}

void Decompressor::ThrowIfCanceled()
{
  if (canceled)
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Operation canceled by user")));
}
